package io.github.cottonmc.cotton.gui.impl;

import com.mojang.datafixers.util.Unit;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.Decoder;
import com.mojang.serialization.Encoder;
import com.mojang.serialization.Lifecycle;
import com.mojang.serialization.MapCodec;
import net.fabricmc.fabric.api.event.Event;
import net.fabricmc.fabric.api.event.EventFactory;
import net.fabricmc.fabric.api.networking.v1.PayloadTypeRegistry;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_2505;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2960;
import net.minecraft.class_5455;
import net.minecraft.class_6903;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import io.github.cottonmc.cotton.gui.SyncedGuiDescription;
import io.github.cottonmc.cotton.gui.networking.NetworkSide;
import io.github.cottonmc.cotton.gui.networking.ScreenMessageKey;
import io.github.cottonmc.cotton.gui.networking.ScreenNetworking;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Executor;

public class ScreenNetworkingImpl implements ScreenNetworking {
	// Matches the one used in PacketCodecs.codec() etc
	private static final long MAX_NBT_SIZE = 0x200000L;
	public static final ScreenMessageKey<Unit> CLIENT_READY_MESSAGE_KEY = new ScreenMessageKey<>(
		LibGuiCommon.id("client_ready"),
		MapCodec.unitCodec(Unit.INSTANCE)
	);

	public record ScreenMessage(int syncId, class_2960 message, class_2520 nbt) implements class_8710 {
		public static final class_9154<ScreenMessage> ID = new class_9154<>(LibGuiCommon.id("screen_message"));
		public static final class_9139<class_9129, ScreenMessage> CODEC = class_9139.method_56436(
			class_9135.field_49675, ScreenMessage::syncId,
			class_2960.field_48267, ScreenMessage::message,
			class_9135.method_56378(() -> class_2505.method_53899(MAX_NBT_SIZE)), ScreenMessage::nbt,
			ScreenMessage::new
		);

		@Override
		public class_9154<? extends class_8710> method_56479() {
			return ID;
		}
	}

	private static final Logger LOGGER = LoggerFactory.getLogger(ScreenNetworkingImpl.class);

	private final Map<class_2960, ReceiverData<?>> receivers = new HashMap<>();
	private final SyncedGuiDescription description;
	private final NetworkSide side;
	private final Event<ReadyListener> readyEvent;
	private boolean ready = false;

	public ScreenNetworkingImpl(SyncedGuiDescription description, NetworkSide side) {
		this.description = description;
		this.side = side;
		this.readyEvent = EventFactory.createArrayBacked(ReadyListener.class, listeners -> screenNetworking -> {
			for (ReadyListener listener : listeners) {
				listener.onConnected(screenNetworking);
			}
		});

		if (side == NetworkSide.SERVER) {
			receive(CLIENT_READY_MESSAGE_KEY, data -> markReady());
		}
	}

	private static class_6903<class_2520> getRegistryOps(class_5455 registryManager) {
		return registryManager.method_57093(class_2509.field_11560);
	}

	@Override
	public <D> void receive(class_2960 message, Decoder<D> decoder, MessageReceiver<D> receiver) {
		Objects.requireNonNull(message, "message");
		Objects.requireNonNull(decoder, "decoder");
		Objects.requireNonNull(receiver, "receiver");

		if (!receivers.containsKey(message)) {
			receivers.put(message, new ReceiverData<>(decoder, receiver));
		} else {
			throw new IllegalStateException("Message " + message + " on side " + side + " already registered");
		}
	}

	@Override
	public <D> void send(class_2960 message, Encoder<D> encoder, D data) {
		Objects.requireNonNull(message, "message");
		Objects.requireNonNull(encoder, "encoder");

		var ops = getRegistryOps(description.getWorld().method_30349());
		class_2520 encoded = encoder.encodeStart(ops, data).getOrThrow();
		ScreenMessage packet = new ScreenMessage(description.field_7763, message, encoded);
		description.getPacketSender().sendPacket(packet);
	}

	@Override
	public Event<ReadyListener> getReadyEvent() {
		return readyEvent;
	}

	public boolean isReady() {
		return ready;
	}

	public void markReady() {
		ready = true;
		getReadyEvent().invoker().onConnected(this);
	}

	public static void init() {
		PayloadTypeRegistry.playS2C().register(ScreenMessage.ID, ScreenMessage.CODEC);
		PayloadTypeRegistry.playC2S().register(ScreenMessage.ID, ScreenMessage.CODEC);
		ServerPlayNetworking.registerGlobalReceiver(ScreenMessage.ID, (payload, context) -> {
			handle(context.server(), context.player(), payload);
		});
	}

	public static void handle(Executor executor, class_1657 player, ScreenMessage packet) {
		class_1703 screenHandler = player.field_7512;

		if (!(screenHandler instanceof SyncedGuiDescription guiDescription)) {
			LOGGER.error("Received message packet for screen handler {} which is not a SyncedGuiDescription", screenHandler);
			return;
		} else if (packet.syncId() != screenHandler.field_7763) {
			LOGGER.error("Received message for sync ID {}, current sync ID: {}", packet.syncId(), screenHandler.field_7763);
			return;
		}

		var networking = (ScreenNetworkingImpl) guiDescription.getNetworking(guiDescription.getNetworkSide());
		ReceiverData<?> receiverData = networking.receivers.get(packet.message());
		if (receiverData != null) {
			processMessage(executor, player, packet, screenHandler, receiverData);
		} else {
			LOGGER.error("Message {} not registered for {} on side {}", packet.message(), screenHandler, networking.side);
		}
	}

	private static <D> void processMessage(Executor executor, class_1657 player, ScreenMessage packet, class_1703 description, ReceiverData<D> receiverData) {
		var ops = getRegistryOps(player.method_56673());
		var result = receiverData.decoder().parse(ops, packet.nbt());

		switch (result) {
			case DataResult.Success(D data, Lifecycle lifecycle) -> executor.execute(() -> {
				try {
					receiverData.receiver().onMessage(data);
				} catch (Exception e) {
					LOGGER.error("Error handling screen message {} for {}", packet.message(), description, e);
				}
			});

			case DataResult.Error<D> error -> LOGGER.error(
				"Could not parse screen message {}: {}",
				packet.message(),
				error.message()
			);
		}
	}

	private record ReceiverData<D>(Decoder<D> decoder, MessageReceiver<D> receiver) {
	}

	public static final class DummyNetworking extends ScreenNetworkingImpl {
		public DummyNetworking() {
			super(null, null);
		}

		@Override
		public <D> void receive(class_2960 message, Decoder<D> decoder, MessageReceiver<D> receiver) {
			// NO-OP
		}

		@Override
		public <D> void send(class_2960 message, Encoder<D> encoder, D data) {
			// NO-OP
		}
	}
}
